<?php
/**
 * Production Maintenance Script
 * 
 * This script handles routine maintenance tasks for the production
 * crypto exchange including monitoring, backups, and health checks.
 */

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/config/production_config.php';
require_once __DIR__ . '/api/blockchain_manager.php';
require_once __DIR__ . '/api/security_validator.php';

class ProductionMaintenance {
    private $db;
    private $blockchain;
    private $config;
    
    public function __construct() {
        $this->db = new Database();
        $this->blockchain = new BlockchainManager($this->db);
        $this->config = ProductionConfig::class;
        
        echo "🔧 YELLOW Crypto Exchange - Production Maintenance\n";
        echo "==================================================\n\n";
    }
    
    /**
     * Run all maintenance tasks
     */
    public function runMaintenance() {
        try {
            echo "🕐 Starting maintenance at " . date('Y-m-d H:i:s') . "\n\n";
            
            // 1. System Health Check
            $this->systemHealthCheck();
            
            // 2. Blockchain Synchronization
            $this->blockchainSync();
            
            // 3. Security Monitoring
            $this->securityMonitoring();
            
            // 4. Database Maintenance
            $this->databaseMaintenance();
            
            // 5. Backup Operations
            $this->backupOperations();
            
            // 6. Performance Monitoring
            $this->performanceMonitoring();
            
            // 7. Alert Processing
            $this->processAlerts();
            
            echo "\n✅ Maintenance completed successfully at " . date('Y-m-d H:i:s') . "\n";
            
        } catch (Exception $e) {
            echo "\n❌ Maintenance failed: " . $e->getMessage() . "\n";
            $this->sendAlert('system_error', 'critical', 'Maintenance failed: ' . $e->getMessage());
        }
    }
    
    /**
     * System health check
     */
    private function systemHealthCheck() {
        echo "🏥 System Health Check...\n";
        
        $conn = $this->db->getConnection();
        $health_score = 100;
        $issues = [];
        
        // Check database connectivity
        try {
            $stmt = $conn->prepare("SELECT 1");
            $stmt->execute();
            echo "   ✅ Database: Connected\n";
        } catch (Exception $e) {
            $health_score -= 30;
            $issues[] = "Database connection failed";
            echo "   ❌ Database: Connection failed\n";
        }
        
        // Check system wallet balances
        $stmt = $conn->prepare("
            SELECT currency, network, balance, min_balance 
            FROM crypto_wallets 
            WHERE user_id = 0 AND is_real_blockchain = 1
        ");
        $stmt->execute();
        $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        $low_balance_wallets = [];
        foreach ($wallets as $wallet) {
            if ($wallet['balance'] < $wallet['min_balance']) {
                $low_balance_wallets[] = $wallet['currency'];
                $health_score -= 10;
            }
        }
        
        if (!empty($low_balance_wallets)) {
            $issues[] = "Low system wallet balances: " . implode(', ', $low_balance_wallets);
            echo "   ⚠️ System Wallets: Low balances detected\n";
        } else {
            echo "   ✅ System Wallets: All balances healthy\n";
        }
        
        // Check transaction volume
        $stmt = $conn->prepare("
            SELECT COUNT(*) as count, SUM(amount) as volume 
            FROM crypto_transactions 
            WHERE created_at >= datetime('now', '-1 hour')
        ");
        $stmt->execute();
        $hourly_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo "   📊 Transactions (last hour): {$hourly_stats['count']} txns, {$hourly_stats['volume']} volume\n";
        
        // Update health score
        $this->updateHealthScore($health_score);
        
        if ($health_score < 80) {
            $this->sendAlert('system_health', 'warning', 'System health score below 80%: ' . implode('; ', $issues));
        }
        
        echo "   📈 System Health Score: $health_score%\n\n";
    }
    
    /**
     * Blockchain synchronization
     */
    private function blockchainSync() {
        echo "⛓️ Blockchain Synchronization...\n";
        
        try {
            $result = $this->blockchain->syncBlockchainBalances();
            
            if ($result['success']) {
                echo "   ✅ Blockchain sync successful\n";
                echo "   📊 Updated {$result['updated_wallets']} wallets\n";
                
                // Log sync success
                $this->logMetric('blockchain_sync', 100, 'healthy');
            } else {
                echo "   ❌ Blockchain sync failed: {$result['error']}\n";
                $this->sendAlert('blockchain_sync', 'error', 'Blockchain sync failed: ' . $result['error']);
                $this->logMetric('blockchain_sync', 0, 'failed');
            }
            
        } catch (Exception $e) {
            echo "   ❌ Blockchain sync error: " . $e->getMessage() . "\n";
            $this->sendAlert('blockchain_sync', 'critical', 'Blockchain sync error: ' . $e->getMessage());
        }
        
        echo "\n";
    }
    
    /**
     * Security monitoring
     */
    private function securityMonitoring() {
        echo "🛡️ Security Monitoring...\n";
        
        $conn = $this->db->getConnection();
        
        // Check for suspicious transactions
        $stmt = $conn->prepare("
            SELECT COUNT(*) as count 
            FROM crypto_transactions 
            WHERE created_at >= datetime('now', '-1 hour') 
            AND amount > 10000
        ");
        $stmt->execute();
        $large_transactions = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        if ($large_transactions > 5) {
            $this->sendAlert('security', 'warning', "High number of large transactions detected: $large_transactions");
            echo "   ⚠️ Large transactions: $large_transactions in last hour\n";
        } else {
            echo "   ✅ Large transactions: Normal activity\n";
        }
        
        // Check for failed login attempts
        $stmt = $conn->prepare("
            SELECT COUNT(*) as count 
            FROM security_logs 
            WHERE event_type = 'failed_login' 
            AND created_at >= datetime('now', '-1 hour')
        ");
        $stmt->execute();
        $failed_logins = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        if ($failed_logins > 10) {
            $this->sendAlert('security', 'warning', "High number of failed login attempts: $failed_logins");
            echo "   ⚠️ Failed logins: $failed_logins in last hour\n";
        } else {
            echo "   ✅ Failed logins: Normal activity\n";
        }
        
        // Check multi-signature requirements
        $stmt = $conn->prepare("
            SELECT COUNT(*) as count 
            FROM multisig_transactions 
            WHERE status = 'pending' 
            AND created_at < datetime('now', '-30 minutes')
        ");
        $stmt->execute();
        $pending_multisig = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        if ($pending_multisig > 0) {
            echo "   ⚠️ Pending multi-sig transactions: $pending_multisig\n";
        } else {
            echo "   ✅ Multi-sig transactions: None pending\n";
        }
        
        echo "\n";
    }
    
    /**
     * Database maintenance
     */
    private function databaseMaintenance() {
        echo "💾 Database Maintenance...\n";
        
        $conn = $this->db->getConnection();
        
        // Clean up old logs (keep last 30 days)
        $stmt = $conn->prepare("
            DELETE FROM security_logs 
            WHERE created_at < datetime('now', '-30 days')
        ");
        $stmt->execute();
        $deleted_security_logs = $stmt->rowCount();
        
        $stmt = $conn->prepare("
            DELETE FROM audit_logs 
            WHERE created_at < datetime('now', '-30 days')
        ");
        $stmt->execute();
        $deleted_audit_logs = $stmt->rowCount();
        
        echo "   🧹 Cleaned up $deleted_security_logs security logs\n";
        echo "   🧹 Cleaned up $deleted_audit_logs audit logs\n";
        
        // Optimize database
        $conn->exec("VACUUM");
        $conn->exec("ANALYZE");
        
        echo "   ✅ Database optimized\n\n";
    }
    
    /**
     * Backup operations
     */
    private function backupOperations() {
        echo "💾 Backup Operations...\n";
        
        $backup_dir = __DIR__ . '/backups/' . date('Y-m-d');
        if (!is_dir($backup_dir)) {
            mkdir($backup_dir, 0755, true);
        }
        
        // Database backup
        $db_backup_file = $backup_dir . '/database_' . date('H-i-s') . '.sqlite';
        if (copy(__DIR__ . '/database/crypto_exchange.db', $db_backup_file)) {
            echo "   ✅ Database backup created: " . basename($db_backup_file) . "\n";
            
            // Log backup
            $this->logBackup('database', $db_backup_file, filesize($db_backup_file), 'success');
        } else {
            echo "   ❌ Database backup failed\n";
            $this->logBackup('database', $db_backup_file, 0, 'failed');
        }
        
        // Wallet backup (export addresses only, not private keys)
        $wallet_backup_file = $backup_dir . '/wallets_' . date('H-i-s') . '.json';
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            SELECT currency, network, address, balance, is_real_blockchain 
            FROM crypto_wallets 
            WHERE user_id = 0
        ");
        $stmt->execute();
        $wallets = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (file_put_contents($wallet_backup_file, json_encode($wallets, JSON_PRETTY_PRINT))) {
            echo "   ✅ Wallet backup created: " . basename($wallet_backup_file) . "\n";
            $this->logBackup('wallets', $wallet_backup_file, filesize($wallet_backup_file), 'success');
        } else {
            echo "   ❌ Wallet backup failed\n";
            $this->logBackup('wallets', $wallet_backup_file, 0, 'failed');
        }
        
        // Clean up old backups (keep last 30 days)
        $this->cleanupOldBackups();
        
        echo "\n";
    }
    
    /**
     * Performance monitoring
     */
    private function performanceMonitoring() {
        echo "📊 Performance Monitoring...\n";
        
        $conn = $this->db->getConnection();
        
        // Transaction volume
        $stmt = $conn->prepare("
            SELECT COUNT(*) as count, SUM(amount) as volume 
            FROM crypto_transactions 
            WHERE created_at >= datetime('now', '-24 hours')
        ");
        $stmt->execute();
        $daily_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo "   📈 Daily transactions: {$daily_stats['count']}\n";
        echo "   💰 Daily volume: {$daily_stats['volume']}\n";
        
        // Active users
        $stmt = $conn->prepare("
            SELECT COUNT(DISTINCT user_id) as count 
            FROM crypto_transactions 
            WHERE created_at >= datetime('now', '-24 hours')
        ");
        $stmt->execute();
        $active_users = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        echo "   👥 Active users (24h): $active_users\n";
        
        // System wallet utilization
        $stmt = $conn->prepare("
            SELECT currency, network, balance, 
                   (balance * 100.0 / (SELECT balance FROM crypto_wallets c2 WHERE c2.currency = crypto_wallets.currency AND c2.user_id = 0 AND c2.is_real_blockchain = 1)) as utilization
            FROM crypto_wallets 
            WHERE user_id = 0 AND is_real_blockchain = 1
        ");
        $stmt->execute();
        $wallet_utilization = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        foreach ($wallet_utilization as $wallet) {
            echo "   💳 {$wallet['currency']} ({$wallet['network']}): {$wallet['utilization']}% utilized\n";
        }
        
        // Log performance metrics
        $this->logMetric('transaction_volume', $daily_stats['volume'], 'normal');
        $this->logMetric('active_users', $active_users, 'normal');
        
        echo "\n";
    }
    
    /**
     * Process alerts
     */
    private function processAlerts() {
        echo "🚨 Alert Processing...\n";
        
        $conn = $this->db->getConnection();
        
        // Check for unresolved alerts
        $stmt = $conn->prepare("
            SELECT COUNT(*) as count 
            FROM production_alerts 
            WHERE resolved = 0 AND created_at < datetime('now', '-1 hour')
        ");
        $stmt->execute();
        $unresolved_alerts = $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        
        if ($unresolved_alerts > 0) {
            echo "   ⚠️ $unresolved_alerts unresolved alerts\n";
            
            // Get recent alerts
            $stmt = $conn->prepare("
                SELECT alert_type, severity, message, created_at 
                FROM production_alerts 
                WHERE resolved = 0 
                ORDER BY created_at DESC 
                LIMIT 5
            ");
            $stmt->execute();
            $recent_alerts = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            foreach ($recent_alerts as $alert) {
                echo "     - [{$alert['severity']}] {$alert['alert_type']}: {$alert['message']}\n";
            }
        } else {
            echo "   ✅ No unresolved alerts\n";
        }
        
        echo "\n";
    }
    
    /**
     * Helper methods
     */
    private function updateHealthScore($score) {
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            UPDATE production_monitoring 
            SET metric_value = ?, status = ?
            WHERE metric_name = 'system_health'
        ");
        
        $status = $score >= 90 ? 'healthy' : ($score >= 70 ? 'warning' : 'critical');
        $stmt->execute([$score, $status]);
    }
    
    private function logMetric($name, $value, $status) {
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            INSERT INTO production_monitoring (metric_name, metric_value, status)
            VALUES (?, ?, ?)
        ");
        $stmt->execute([$name, $value, $status]);
    }
    
    private function sendAlert($type, $severity, $message) {
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            INSERT INTO production_alerts (alert_type, severity, message)
            VALUES (?, ?, ?)
        ");
        $stmt->execute([$type, $severity, $message]);
        
        // In production, also send email/SMS
        echo "   🚨 ALERT [$severity]: $message\n";
    }
    
    private function logBackup($type, $file_path, $file_size, $status) {
        $conn = $this->db->getConnection();
        $stmt = $conn->prepare("
            INSERT INTO backup_logs (backup_type, file_path, file_size, status)
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([$type, $file_path, $file_size, $status]);
    }
    
    private function cleanupOldBackups() {
        $backup_root = __DIR__ . '/backups';
        $days_to_keep = 30;
        
        if (is_dir($backup_root)) {
            $directories = glob($backup_root . '/*', GLOB_ONLYDIR);
            
            foreach ($directories as $dir) {
                $dir_time = filemtime($dir);
                if ($dir_time < (time() - ($days_to_keep * 24 * 60 * 60))) {
                    $this->deleteDirectory($dir);
                    echo "   🗑️ Removed old backup: " . basename($dir) . "\n";
                }
            }
        }
    }
    
    private function deleteDirectory($dir) {
        if (is_dir($dir)) {
            $files = array_diff(scandir($dir), ['.', '..']);
            foreach ($files as $file) {
                unlink($dir . '/' . $file);
            }
            rmdir($dir);
        }
    }
}

// ========================================
// EXECUTION
// ========================================

if (php_sapi_name() === 'cli') {
    $maintenance = new ProductionMaintenance();
    $maintenance->runMaintenance();
} else {
    echo "This script must be run from the command line.\n";
    echo "Usage: php production_maintenance.php\n";
}
?>

