<?php
/**
 * Replace System Wallets with Your Own Wallets
 * 
 * This script allows you to replace the generated system wallets
 * with your own wallet addresses and private keys that you control.
 * 
 * ⚠️ SECURITY WARNING:
 * - Only use wallets you fully control
 * - Never share your private keys
 * - Test with small amounts first
 * - Delete this file after use
 */

require_once 'config/env.php';
require_once 'config/database.php';

class SystemWalletReplacer {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
        EnvConfig::load(__DIR__ . '/config.env');
    }
    
    /**
     * Replace system wallets with your own
     */
    public function replaceSystemWallets() {
        echo "🔄 SYSTEM WALLET REPLACEMENT TOOL\n";
        echo "==================================\n\n";
        
        echo "This tool will help you replace the generated system wallets\n";
        echo "with your own wallet addresses and private keys.\n\n";
        
        echo "⚠️  IMPORTANT:\n";
        echo "- Only use wallets you fully control\n";
        echo "- Make sure you have the private keys\n";
        echo "- Test with small amounts first\n";
        echo "- Backup your current system before proceeding\n\n";
        
        // Get current system wallets
        $currentWallets = $this->getCurrentSystemWallets();
        
        echo "📋 CURRENT SYSTEM WALLETS:\n";
        echo "==========================\n";
        foreach ($currentWallets as $wallet) {
            echo "• {$wallet['currency']} ({$wallet['network']}): {$wallet['address']}\n";
            echo "  Balance: {$wallet['balance']} {$wallet['currency']}\n";
            echo "  USD Value: $" . number_format($wallet['usd_value'], 2) . "\n\n";
        }
        
        echo "🔄 REPLACEMENT OPTIONS:\n";
        echo "=======================\n";
        echo "1. Replace individual wallets\n";
        echo "2. Replace all wallets at once\n";
        echo "3. Add new wallet types\n";
        echo "4. View replacement instructions\n";
        echo "5. Exit\n\n";
        
        $choice = readline("Choose an option (1-5): ");
        
        switch ($choice) {
            case '1':
                $this->replaceIndividualWallets();
                break;
            case '2':
                $this->replaceAllWallets();
                break;
            case '3':
                $this->addNewWalletTypes();
                break;
            case '4':
                $this->showInstructions();
                break;
            case '5':
                echo "Exiting...\n";
                break;
            default:
                echo "Invalid option. Please try again.\n";
                $this->replaceSystemWallets();
        }
    }
    
    /**
     * Replace individual wallets
     */
    private function replaceIndividualWallets() {
        echo "\n🔄 REPLACE INDIVIDUAL WALLETS\n";
        echo "=============================\n\n";
        
        $currentWallets = $this->getCurrentSystemWallets();
        
        foreach ($currentWallets as $wallet) {
            echo "Replace {$wallet['currency']} ({$wallet['network']}) wallet?\n";
            echo "Current: {$wallet['address']}\n";
            echo "Balance: {$wallet['balance']} {$wallet['currency']}\n\n";
            
            $replace = readline("Replace this wallet? (y/n): ");
            
            if (strtolower($replace) === 'y') {
                $this->replaceSingleWallet($wallet);
            }
        }
    }
    
    /**
     * Replace a single wallet
     */
    private function replaceSingleWallet($wallet) {
        echo "\n📝 ENTER NEW WALLET DETAILS:\n";
        echo "=============================\n";
        
        $newAddress = readline("New wallet address: ");
        $newPrivateKey = readline("New private key: ");
        
        if (empty($newAddress) || empty($newPrivateKey)) {
            echo "❌ Address and private key are required!\n";
            return;
        }
        
        // Validate address format
        if (!$this->validateAddress($newAddress, $wallet['network'])) {
            echo "❌ Invalid address format for {$wallet['network']} network!\n";
            return;
        }
        
        // Encrypt the private key
        $encryptedKey = $this->encryptPrivateKey($newPrivateKey);
        
        // Update the wallet in database
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                UPDATE crypto_wallets 
                SET address = ?, private_key = ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            ");
            $stmt->execute([$newAddress, $encryptedKey, $wallet['id']]);
            
            // Update production config
            $this->updateProductionConfig($wallet['currency'], $wallet['network'], $newAddress, $encryptedKey);
            
            echo "✅ Wallet replaced successfully!\n";
            echo "New address: {$newAddress}\n\n";
            
        } catch (Exception $e) {
            echo "❌ Error replacing wallet: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * Replace all wallets at once
     */
    private function replaceAllWallets() {
        echo "\n🔄 REPLACE ALL WALLETS\n";
        echo "=====================\n\n";
        
        echo "⚠️  WARNING: This will replace ALL system wallets!\n";
        echo "Make sure you have all the new wallet details ready.\n\n";
        
        $confirm = readline("Are you sure you want to replace all wallets? (yes/no): ");
        
        if (strtolower($confirm) !== 'yes') {
            echo "Operation cancelled.\n";
            return;
        }
        
        $currentWallets = $this->getCurrentSystemWallets();
        
        foreach ($currentWallets as $wallet) {
            echo "\n--- {$wallet['currency']} ({$wallet['network']}) ---\n";
            $newAddress = readline("New address: ");
            $newPrivateKey = readline("New private key: ");
            
            if (!empty($newAddress) && !empty($newPrivateKey)) {
                $this->replaceSingleWallet($wallet);
            } else {
                echo "Skipping {$wallet['currency']} wallet...\n";
            }
        }
    }
    
    /**
     * Add new wallet types
     */
    private function addNewWalletTypes() {
        echo "\n➕ ADD NEW WALLET TYPES\n";
        echo "======================\n\n";
        
        $currency = readline("Currency (e.g., BTC, ETH, USDT): ");
        $network = readline("Network (e.g., ERC20, BEP20, BTC): ");
        $address = readline("Wallet address: ");
        $privateKey = readline("Private key: ");
        
        if (empty($currency) || empty($network) || empty($address) || empty($privateKey)) {
            echo "❌ All fields are required!\n";
            return;
        }
        
        // Validate address format
        if (!$this->validateAddress($address, $network)) {
            echo "❌ Invalid address format for {$network} network!\n";
            return;
        }
        
        // Encrypt the private key
        $encryptedKey = $this->encryptPrivateKey($privateKey);
        
        // Add to database
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                INSERT INTO crypto_wallets (user_id, network, currency, address, private_key, balance, is_real_blockchain, created_at, updated_at)
                VALUES (0, ?, ?, ?, ?, 0, 1, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
            ");
            $stmt->execute([$network, $currency, $address, $encryptedKey]);
            
            echo "✅ New wallet type added successfully!\n";
            echo "Currency: {$currency}\n";
            echo "Network: {$network}\n";
            echo "Address: {$address}\n\n";
            
        } catch (Exception $e) {
            echo "❌ Error adding wallet: " . $e->getMessage() . "\n";
        }
    }
    
    /**
     * Show replacement instructions
     */
    private function showInstructions() {
        echo "\n📖 REPLACEMENT INSTRUCTIONS\n";
        echo "===========================\n\n";
        
        echo "1. PREPARATION:\n";
        echo "   - Have your wallet addresses ready\n";
        echo "   - Have your private keys ready\n";
        echo "   - Test addresses on blockchain explorers\n";
        echo "   - Backup current system wallets\n\n";
        
        echo "2. SECURITY:\n";
        echo "   - Use wallets you fully control\n";
        echo "   - Never share private keys\n";
        echo "   - Use hardware wallets if possible\n";
        echo "   - Test with small amounts first\n\n";
        
        echo "3. SUPPORTED NETWORKS:\n";
        echo "   - ERC20 (Ethereum): 0x... addresses\n";
        echo "   - BEP20 (BSC): 0x... addresses\n";
        echo "   - TRC20 (Tron): T... addresses\n";
        echo "   - BTC (Bitcoin): 1... or 3... addresses\n";
        echo "   - SOL (Solana): Base58 addresses\n\n";
        
        echo "4. AFTER REPLACEMENT:\n";
        echo "   - Fund the new wallets\n";
        echo "   - Test transactions\n";
        echo "   - Monitor balances\n";
        echo "   - Update backup procedures\n\n";
        
        echo "5. EXAMPLE WALLET ADDRESSES:\n";
        echo "   - USDT (ERC20): 0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6\n";
        echo "   - USDT (BEP20): 0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6\n";
        echo "   - USDT (TRC20): TQn9Y2khEsLJW1ChVWFMSMeRDow5KcbLSE\n";
        echo "   - BTC: 1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa\n";
        echo "   - SOL: So1b118851539a77da781a62be5a8d9abcbebae2bf\n\n";
    }
    
    /**
     * Get current system wallets
     */
    private function getCurrentSystemWallets() {
        try {
            $conn = $this->db->getConnection();
            $stmt = $conn->prepare("
                SELECT 
                    cw.id, cw.network, cw.currency, cw.address, cw.balance,
                    (cw.balance * COALESCE(cp.buy_price, 0)) as usd_value
                FROM crypto_wallets cw
                LEFT JOIN crypto_prices cp ON cw.currency = cp.crypto
                WHERE cw.user_id = 0
                ORDER BY cw.currency, cw.network
            ");
            $stmt->execute();
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error getting system wallets: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Validate address format
     */
    private function validateAddress($address, $network) {
        $patterns = [
            'erc20' => '/^0x[a-fA-F0-9]{40}$/',
            'bep20' => '/^0x[a-fA-F0-9]{40}$/',
            'polygon' => '/^0x[a-fA-F0-9]{40}$/',
            'trc20' => '/^T[a-zA-Z0-9]{33}$/',
            'btc' => '/^[13][a-km-zA-HJ-NP-Z1-9]{25,34}$/',
            'solana' => '/^[1-9A-HJ-NP-Za-km-z]{32,44}$/'
        ];
        
        $network_lower = strtolower($network);
        return isset($patterns[$network_lower]) && preg_match($patterns[$network_lower], $address);
    }
    
    /**
     * Encrypt private key
     */
    private function encryptPrivateKey($privateKey) {
        $jwtSecret = EnvConfig::get('JWT_SECRET');
        $hash = hash('sha256', $jwtSecret);
        return base64_encode($privateKey . '|' . $hash);
    }
    
    /**
     * Update production config
     */
    private function updateProductionConfig($currency, $network, $address, $encryptedKey) {
        $configFile = 'config/production_config.php';
        $content = file_get_contents($configFile);
        
        // Find and replace the specific wallet entry
        $pattern = "/'{$currency}_{$network}' => \[(.*?)\],/s";
        $replacement = "'{$currency}_{$network}' => [
                'address' => '{$address}',
                'private_key_encrypted' => '{$encryptedKey}',
                'currency' => '{$currency}',
                'network' => '{$network}',
                'is_real_blockchain' => true
            ],";
        
        $content = preg_replace($pattern, $replacement, $content);
        file_put_contents($configFile, $content);
        
        echo "✅ Production config updated\n";
    }
}

// Run the replacement tool
$replacer = new SystemWalletReplacer();
$replacer->replaceSystemWallets();
?>
