<?php
/**
 * Database Setup Script
 * Run this script to initialize the database and create initial data
 */

require_once __DIR__ . '/config/database.php';

class DatabaseSetup {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function setup() {
        echo "🚀 Setting up YELLOW Crypto Exchange Database...\n\n";
        
        try {
            // Read and execute schema
            $this->executeSchema();
            
            // Insert initial data
            $this->insertInitialData();
            
            // Create admin user
            $this->createAdminUser();
            
            // Create trading pairs
            $this->createTradingPairs();
            
            // Update market prices
            $this->updateMarketPrices();
            
            echo "✅ Database setup completed successfully!\n";
            echo "📊 Admin Panel: http://localhost:8000/admin/\n";
            echo "🔗 API Endpoint: http://localhost:8000/api/\n";
            echo "🌐 Frontend: http://localhost:8000/\n\n";
            
        } catch (Exception $e) {
            echo "❌ Setup failed: " . $e->getMessage() . "\n";
            exit(1);
        }
    }
    
    private function executeSchema() {
        echo "📋 Creating database schema...\n";
        
        $schema_file = __DIR__ . '/database/schema.sql';
        if (!file_exists($schema_file)) {
            throw new Exception("Schema file not found: $schema_file");
        }
        
        $sql = file_get_contents($schema_file);
        $conn = $this->db->getConnection();
        
        // Split SQL into individual statements
        $statements = array_filter(array_map('trim', explode(';', $sql)));
        
        foreach ($statements as $statement) {
            if (!empty($statement)) {
                $conn->exec($statement);
            }
        }
        
        echo "✅ Database schema created successfully\n";
    }
    
    private function insertInitialData() {
        echo "📊 Inserting initial data...\n";
        
        $conn = $this->db->getConnection();
        
        // Insert cryptocurrencies if not exists
        $cryptos = [
            ['USDT', 'Tether', 'https://lh3.googleusercontent.com/aida-public/AB6AXuBjeal1Ll7qsNNfmNAz3K1871s0o4k4Z9deMRTZiv8bcbt_CkHU8Nt2jpv3ezWmNRl7quWNnsmJM132FKeSZQ1crYBObrpQ4ZxVhaWYK35N5rBXNW6W_oGgO6t1pE01KUkaiSi5hFiA8LIbYXw5brx6fVUtorwgtUXyL7sEtDtWF7P2CUz-oD9_M5iCbGXF1m16hIB_ZIixWR9URiIEWc0Ma_sLUqKiy-lZqXsmAYo6ioVsvrsmAnC8mco1BwNpWtV3VjuF3jgbsl4'],
            ['USDC', 'USD Coin', 'https://lh3.googleusercontent.com/aida-public/AB6AXuBjeal1Ll7qsNNfmNAz3K1871s0o4k4Z9deMRTZiv8bcbt_CkHU8Nt2jpv3ezWmNRl7quWNnsmJM132FKeSZQ1crYBObrpQ4ZxVhaWYK35N5rBXNW6W_oGgO6t1pE01KUkaiSi5hFiA8LIbYXw5brx6fVUtorwgtUXyL7sEtDtWF7P2CUz-oD9_M5iCbGXF1m16hIB_ZIixWR9URiIEWc0Ma_sLUqKiy-lZqXsmAYo6ioVsvrsmAnC8mco1BwNpWtV3VjuF3jgbsl4'],
            ['BTC', 'Bitcoin', 'https://cryptologos.cc/logos/bitcoin-btc-logo.png'],
            ['ETH', 'Ethereum', 'https://cryptologos.cc/logos/ethereum-eth-logo.png'],
            ['BNB', 'Binance Coin', 'https://cryptologos.cc/logos/bnb-bnb-logo.png'],
            ['ADA', 'Cardano', 'https://cryptologos.cc/logos/cardano-ada-logo.png'],
            ['SOL', 'Solana', 'https://cryptologos.cc/logos/solana-sol-logo.png'],
            ['MATIC', 'Polygon', 'https://cryptologos.cc/logos/polygon-matic-logo.png']
        ];
        
        foreach ($cryptos as $crypto) {
            $stmt = $conn->prepare("
                INSERT IGNORE INTO cryptocurrencies (symbol, name, logo_url) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute($crypto);
        }
        
        // Insert networks if not exists
        $networks = [
            ['Ethereum', 'ERC20', 'public', 2.50],
            ['Tron', 'TRC20', 'account_balance', 1.00],
            ['BSC', 'BEP20', 'hub', 0.50],
            ['Polygon', 'MATIC', 'hexagon', 0.20],
            ['Base', 'BASE', 'layers', 0.30],
            ['Solana', 'SOL', 'bolt', 0.01]
        ];
        
        foreach ($networks as $network) {
            $stmt = $conn->prepare("
                INSERT IGNORE INTO networks (name, symbol, icon, fee_usd) 
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute($network);
        }
        
        // Insert fiat currencies if not exists
        $fiats = [
            ['UGX', 'Ugandan Shilling', 'USh', 'UG'],
            ['KES', 'Kenyan Shilling', 'KSh', 'KE'],
            ['TZS', 'Tanzanian Shilling', 'TSh', 'TZ'],
            ['RWF', 'Rwandan Franc', 'RF', 'RW'],
            ['CDF', 'Congolese Franc', 'FC', 'CD'],
            ['NGN', 'Nigerian Naira', '₦', 'NG'],
            ['USD', 'US Dollar', '$', 'US'],
            ['EUR', 'Euro', '€', 'EU'],
            ['GBP', 'British Pound', '£', 'GB']
        ];
        
        foreach ($fiats as $fiat) {
            $stmt = $conn->prepare("
                INSERT IGNORE INTO fiat_currencies (code, name, symbol, country) 
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute($fiat);
        }
        
        echo "✅ Initial data inserted successfully\n";
    }
    
    private function createAdminUser() {
        echo "👤 Creating admin user...\n";
        
        $conn = $this->db->getConnection();
        
        // Check if admin already exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute(['admin@yellowcrypto.com']);
        
        if ($stmt->fetch()) {
            echo "ℹ️  Admin user already exists\n";
            return;
        }
        
        // Create admin user
        $password_hash = password_hash('admin123', PASSWORD_BCRYPT, ['cost' => BCRYPT_ROUNDS]);
        $referral_code = strtoupper(substr(md5(uniqid()), 0, 8));
        
        $stmt = $conn->prepare("
            INSERT INTO users (email, password_hash, first_name, last_name, country, currency, referral_code, is_verified, kyc_status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            'admin@yellowcrypto.com',
            $password_hash,
            'Admin',
            'User',
            'UG',
            'UGX',
            $referral_code,
            1,
            'approved'
        ]);
        
        echo "✅ Admin user created (email: admin@yellowcrypto.com, password: admin123)\n";
    }
    
    private function createTradingPairs() {
        echo "💱 Creating trading pairs...\n";
        
        $conn = $this->db->getConnection();
        
        // Get crypto IDs
        $stmt = $conn->prepare("SELECT id, symbol FROM cryptocurrencies WHERE symbol IN ('USDT', 'USDC', 'BTC', 'ETH')");
        $stmt->execute();
        $cryptos = $stmt->fetchAll();
        
        $crypto_map = [];
        foreach ($cryptos as $crypto) {
            $crypto_map[$crypto['symbol']] = $crypto['id'];
        }
        
        // Get fiat currency IDs
        $stmt = $conn->prepare("SELECT id, code FROM fiat_currencies WHERE code IN ('UGX', 'KES', 'TZS', 'RWF', 'CDF', 'NGN', 'USD')");
        $stmt->execute();
        $fiats = $stmt->fetchAll();
        
        $fiat_map = [];
        foreach ($fiats as $fiat) {
            $fiat_map[$fiat['code']] = $fiat['id'];
        }
        
        // Create trading pairs
        $pairs = [
            // USDT pairs
            ['USDT', 'UGX'],
            ['USDT', 'KES'],
            ['USDT', 'TZS'],
            ['USDT', 'RWF'],
            ['USDT', 'CDF'],
            ['USDT', 'NGN'],
            ['USDT', 'USD'],
            
            // USDC pairs
            ['USDC', 'UGX'],
            ['USDC', 'KES'],
            ['USDC', 'USD'],
            
            // BTC pairs
            ['BTC', 'USDT'],
            ['BTC', 'USD'],
            
            // ETH pairs
            ['ETH', 'USDT'],
            ['ETH', 'USD']
        ];
        
        foreach ($pairs as $pair) {
            $base_symbol = $pair[0];
            $quote_symbol = $pair[1];
            
            if (isset($crypto_map[$base_symbol]) && isset($crypto_map[$quote_symbol])) {
                $base_id = $crypto_map[$base_symbol];
                $quote_id = $crypto_map[$quote_symbol];
                $symbol = $base_symbol . '/' . $quote_symbol;
                
                $stmt = $conn->prepare("
                    INSERT IGNORE INTO trading_pairs (base_crypto_id, quote_crypto_id, symbol) 
                    VALUES (?, ?, ?)
                ");
                $stmt->execute([$base_id, $quote_id, $symbol]);
            }
        }
        
        echo "✅ Trading pairs created successfully\n";
    }
    
    private function updateMarketPrices() {
        echo "📈 Updating market prices...\n";
        
        $conn = $this->db->getConnection();
        
        // Get all trading pairs
        $stmt = $conn->prepare("SELECT id, symbol FROM trading_pairs WHERE is_active = 1");
        $stmt->execute();
        $pairs = $stmt->fetchAll();
        
        foreach ($pairs as $pair) {
            // Generate mock price data
            $price_data = $this->generateMockPrice($pair['symbol']);
            
            if ($price_data) {
                $stmt = $conn->prepare("
                    INSERT INTO market_prices (pair_id, price, volume_24h, change_24h, high_24h, low_24h)
                    VALUES (?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                    price = VALUES(price),
                    volume_24h = VALUES(volume_24h),
                    change_24h = VALUES(change_24h),
                    high_24h = VALUES(high_24h),
                    low_24h = VALUES(low_24h),
                    updated_at = CURRENT_TIMESTAMP
                ");
                $stmt->execute([
                    $pair['id'],
                    $price_data['price'],
                    $price_data['volume_24h'],
                    $price_data['change_24h'],
                    $price_data['high_24h'],
                    $price_data['low_24h']
                ]);
            }
        }
        
        echo "✅ Market prices updated successfully\n";
    }
    
    private function generateMockPrice($symbol) {
        $base_prices = [
            'USDT' => 1.00,
            'USDC' => 1.00,
            'BTC' => 45000.00,
            'ETH' => 3000.00
        ];
        
        $quote_rates = [
            'UGX' => 3700,
            'KES' => 150,
            'TZS' => 2500,
            'RWF' => 1200,
            'CDF' => 2500,
            'NGN' => 800,
            'USD' => 1
        ];
        
        $parts = explode('/', $symbol);
        if (count($parts) !== 2) return null;
        
        $base_symbol = $parts[0];
        $quote_symbol = $parts[1];
        
        if (!isset($base_prices[$base_symbol]) || !isset($quote_rates[$quote_symbol])) {
            return null;
        }
        
        $base_price = $base_prices[$base_symbol];
        $quote_rate = $quote_rates[$quote_symbol];
        $price = $base_price * $quote_rate;
        
        return [
            'price' => $price,
            'volume_24h' => rand(1000000, 10000000),
            'change_24h' => rand(-10, 10) + (rand(0, 99) / 100),
            'high_24h' => $price * (1 + rand(1, 5) / 100),
            'low_24h' => $price * (1 - rand(1, 5) / 100)
        ];
    }
}

// Run setup if called directly
if (basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    $setup = new DatabaseSetup();
    $setup->setup();
}
?>
