<?php
/**
 * Shared Hosting Setup Script
 * Optimized for cPanel shared hosting environments
 */

echo "🚀 YELLOW Crypto Exchange - Shared Hosting Setup\n";
echo "===============================================\n";
echo "Target: https://kako.exchange/\n";
echo "Database: kakoexchange25_kakoexchange\n\n";

// Load shared hosting optimized database configuration
require_once __DIR__ . '/config/database.php';

class SharedHostingSetup {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function setup() {
        echo "🔍 Testing database connection...\n";
        
        try {
            $conn = $this->db->getConnection();
            echo "✅ Database connection successful!\n\n";
            
            // Create tables if they don't exist
            $this->createTables();
            
            // Insert initial data
            $this->insertInitialData();
            
            // Create admin user
            $this->createAdminUser();
            
            echo "✅ Shared hosting setup completed successfully!\n";
            echo "📊 Admin Panel: https://kako.exchange/admin/\n";
            echo "🔗 API Endpoint: https://kako.exchange/api/\n";
            echo "🌐 Frontend: https://kako.exchange/\n\n";
            
        } catch (Exception $e) {
            echo "❌ Setup failed: " . $e->getMessage() . "\n";
            exit(1);
        }
    }
    
    private function createTables() {
        echo "📋 Creating database tables...\n";
        
        $conn = $this->db->getConnection();
        
        // Essential tables for shared hosting
        $tables = [
            // Users table
            "CREATE TABLE IF NOT EXISTS users (
                id INT PRIMARY KEY AUTO_INCREMENT,
                email VARCHAR(255) UNIQUE NOT NULL,
                phone VARCHAR(20) UNIQUE,
                password_hash VARCHAR(255) NOT NULL,
                first_name VARCHAR(100) NOT NULL,
                last_name VARCHAR(100) NOT NULL,
                country VARCHAR(3) NOT NULL DEFAULT 'UG',
                currency VARCHAR(3) NOT NULL DEFAULT 'UGX',
                is_verified BOOLEAN DEFAULT FALSE,
                is_active BOOLEAN DEFAULT TRUE,
                kyc_status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
                referral_code VARCHAR(20) UNIQUE,
                referred_by INT,
                two_factor_enabled BOOLEAN DEFAULT FALSE,
                two_factor_secret VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (referred_by) REFERENCES users(id)
            )",
            
            // Cryptocurrencies table
            "CREATE TABLE IF NOT EXISTS cryptocurrencies (
                id INT PRIMARY KEY AUTO_INCREMENT,
                symbol VARCHAR(10) UNIQUE NOT NULL,
                name VARCHAR(100) NOT NULL,
                logo_url VARCHAR(255),
                is_active BOOLEAN DEFAULT TRUE,
                min_withdrawal DECIMAL(20,8) DEFAULT 0.00000001,
                max_withdrawal DECIMAL(20,8) DEFAULT 1000000.00000000,
                withdrawal_fee DECIMAL(20,8) DEFAULT 0.00000000,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )",
            
            // Networks table
            "CREATE TABLE IF NOT EXISTS networks (
                id INT PRIMARY KEY AUTO_INCREMENT,
                name VARCHAR(50) NOT NULL,
                symbol VARCHAR(20) NOT NULL,
                icon VARCHAR(50),
                fee_usd DECIMAL(10,2) DEFAULT 0.00,
                is_active BOOLEAN DEFAULT TRUE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )",
            
            // User wallets table
            "CREATE TABLE IF NOT EXISTS user_wallets (
                id INT PRIMARY KEY AUTO_INCREMENT,
                user_id INT NOT NULL,
                crypto_id INT NOT NULL,
                network_id INT NOT NULL,
                address VARCHAR(255) NOT NULL,
                private_key_encrypted TEXT,
                balance DECIMAL(20,8) DEFAULT 0.00000000,
                locked_balance DECIMAL(20,8) DEFAULT 0.00000000,
                is_active BOOLEAN DEFAULT TRUE,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
                FOREIGN KEY (crypto_id) REFERENCES cryptocurrencies(id),
                FOREIGN KEY (network_id) REFERENCES networks(id),
                UNIQUE KEY unique_wallet (user_id, crypto_id, network_id)
            )",
            
            // Transactions table
            "CREATE TABLE IF NOT EXISTS transactions (
                id INT PRIMARY KEY AUTO_INCREMENT,
                user_id INT NOT NULL,
                type ENUM('buy', 'sell', 'send', 'receive', 'deposit', 'withdrawal', 'convert') NOT NULL,
                crypto_id INT,
                network_id INT,
                amount DECIMAL(20,8),
                fee DECIMAL(20,8) DEFAULT 0.00000000,
                from_address VARCHAR(255),
                to_address VARCHAR(255),
                tx_hash VARCHAR(255),
                status ENUM('pending', 'confirmed', 'failed', 'cancelled') DEFAULT 'pending',
                confirmation_count INT DEFAULT 0,
                required_confirmations INT DEFAULT 3,
                notes TEXT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id),
                FOREIGN KEY (crypto_id) REFERENCES cryptocurrencies(id),
                FOREIGN KEY (network_id) REFERENCES networks(id)
            )",
            
            // System settings table
            "CREATE TABLE IF NOT EXISTS system_settings (
                id INT PRIMARY KEY AUTO_INCREMENT,
                setting_key VARCHAR(100) UNIQUE NOT NULL,
                setting_value TEXT,
                description TEXT,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
            )"
        ];
        
        foreach ($tables as $table) {
            try {
                $conn->exec($table);
            } catch (PDOException $e) {
                // Ignore "table already exists" errors
                if (strpos($e->getMessage(), 'already exists') === false) {
                    echo "⚠️  Table creation warning: " . $e->getMessage() . "\n";
                }
            }
        }
        
        echo "✅ Database tables created\n";
    }
    
    private function insertInitialData() {
        echo "📊 Inserting initial data...\n";
        
        $conn = $this->db->getConnection();
        
        // Insert cryptocurrencies
        $cryptos = [
            ['USDT', 'Tether', 'https://lh3.googleusercontent.com/aida-public/AB6AXuBjeal1Ll7qsNNfmNAz3K1871s0o4k4Z9deMRTZiv8bcbt_CkHU8Nt2jpv3ezWmNRl7quWNnsmJM132FKeSZQ1crYBObrpQ4ZxVhaWYK35N5rBXNW6W_oGgO6t1pE01KUkaiSi5hFiA8LIbYXw5brx6fVUtorwgtUXyL7sEtDtWF7P2CUz-oD9_M5iCbGXF1m16hIB_ZIixWR9URiIEWc0Ma_sLUqKiy-lZqXsmAYo6ioVsvrsmAnC8mco1BwNpWtV3VjuF3jgbsl4'],
            ['USDC', 'USD Coin', 'https://lh3.googleusercontent.com/aida-public/AB6AXuBjeal1Ll7qsNNfmNAz3K1871s0o4k4Z9deMRTZiv8bcbt_CkHU8Nt2jpv3ezWmNRl7quWNnsmJM132FKeSZQ1crYBObrpQ4ZxVhaWYK35N5rBXNW6W_oGgO6t1pE01KUkaiSi5hFiA8LIbYXw5brx6fVUtorwgtUXyL7sEtDtWF7P2CUz-oD9_M5iCbGXF1m16hIB_ZIixWR9URiIEWc0Ma_sLUqKiy-lZqXsmAYo6ioVsvrsmAnC8mco1BwNpWtV3VjuF3jgbsl4'],
            ['BTC', 'Bitcoin', 'https://cryptologos.cc/logos/bitcoin-btc-logo.png'],
            ['ETH', 'Ethereum', 'https://cryptologos.cc/logos/ethereum-eth-logo.png']
        ];
        
        foreach ($cryptos as $crypto) {
            $stmt = $conn->prepare("
                INSERT IGNORE INTO cryptocurrencies (symbol, name, logo_url) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute($crypto);
        }
        
        // Insert networks
        $networks = [
            ['Ethereum', 'ERC20', 'public', 2.50],
            ['Tron', 'TRC20', 'account_balance', 1.00],
            ['BSC', 'BEP20', 'hub', 0.50],
            ['Polygon', 'MATIC', 'hexagon', 0.20],
            ['Base', 'BASE', 'layers', 0.30],
            ['Solana', 'SOL', 'bolt', 0.01]
        ];
        
        foreach ($networks as $network) {
            $stmt = $conn->prepare("
                INSERT IGNORE INTO networks (name, symbol, icon, fee_usd) 
                VALUES (?, ?, ?, ?)
            ");
            $stmt->execute($network);
        }
        
        echo "✅ Initial data inserted\n";
    }
    
    private function createAdminUser() {
        echo "👤 Creating admin user...\n";
        
        $conn = $this->db->getConnection();
        
        // Check if admin already exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute(['admin@yellowcrypto.com']);
        
        if ($stmt->fetch()) {
            echo "ℹ️  Admin user already exists\n";
            return;
        }
        
        // Create admin user
        $password_hash = password_hash('admin123', PASSWORD_BCRYPT, ['cost' => 10]);
        $referral_code = strtoupper(substr(md5(uniqid()), 0, 8));
        
        $stmt = $conn->prepare("
            INSERT INTO users (email, phone, password_hash, first_name, last_name, country, referral_code, is_verified, kyc_status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            'admin@yellowcrypto.com',
            '+256700000000',
            $password_hash,
            'Admin',
            'User',
            'UG',
            $referral_code,
            1,
            'approved'
        ]);
        
        echo "✅ Admin user created (email: admin@yellowcrypto.com, password: admin123)\n";
    }
}

// Run setup if called directly
if (basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    $setup = new SharedHostingSetup();
    $setup->setup();
}
?>
