<?php
/**
 * SQLite Database Setup Script
 * Creates the database and initial data for YELLOW Crypto Exchange
 */

require_once __DIR__ . '/config/database_local.php';

class SQLiteSetup {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function setup() {
        echo "🚀 Setting up YELLOW Crypto Exchange SQLite Database...\n\n";
        
        try {
            // Database already created by Database class
            echo "✅ Database connection established\n";
            
            // Insert initial data
            $this->insertInitialData();
            
            // Create admin user
            $this->createAdminUser();
            
            // Create sample trading pairs
            $this->createTradingPairs();
            
            // Update market prices
            $this->updateMarketPrices();
            
            echo "✅ Database setup completed successfully!\n";
            echo "📊 Admin Panel: http://localhost:8000/admin/\n";
            echo "🔗 API Endpoint: http://localhost:8000/api/\n";
            echo "🌐 Frontend: http://localhost:8000/\n\n";
            
        } catch (Exception $e) {
            echo "❌ Setup failed: " . $e->getMessage() . "\n";
            exit(1);
        }
    }
    
    private function insertInitialData() {
        echo "📊 Inserting initial data...\n";
        
        $conn = $this->db->getConnection();
        
        // Insert cryptocurrencies
        $cryptos = [
            ['USDT', 'Tether', 'https://lh3.googleusercontent.com/aida-public/AB6AXuBjeal1Ll7qsNNfmNAz3K1871s0o4k4Z9deMRTZiv8bcbt_CkHU8Nt2jpv3ezWmNRl7quWNnsmJM132FKeSZQ1crYBObrpQ4ZxVhaWYK35N5rBXNW6W_oGgO6t1pE01KUkaiSi5hFiA8LIbYXw5brx6fVUtorwgtUXyL7sEtDtWF7P2CUz-oD9_M5iCbGXF1m16hIB_ZIixWR9URiIEWc0Ma_sLUqKiy-lZqXsmAYo6ioVsvrsmAnC8mco1BwNpWtV3VjuF3jgbsl4'],
            ['USDC', 'USD Coin', 'https://lh3.googleusercontent.com/aida-public/AB6AXuBjeal1Ll7qsNNfmNAz3K1871s0o4k4Z9deMRTZiv8bcbt_CkHU8Nt2jpv3ezWmNRl7quWNnsmJM132FKeSZQ1crYBObrpQ4ZxVhaWYK35N5rBXNW6W_oGgO6t1pE01KUkaiSi5hFiA8LIbYXw5brx6fVUtorwgtUXyL7sEtDtWF7P2CUz-oD9_M5iCbGXF1m16hIB_ZIixWR9URiIEWc0Ma_sLUqKiy-lZqXsmAYo6ioVsvrsmAnC8mco1BwNpWtV3VjuF3jgbsl4'],
            ['BTC', 'Bitcoin', 'https://cryptologos.cc/logos/bitcoin-btc-logo.png'],
            ['ETH', 'Ethereum', 'https://cryptologos.cc/logos/ethereum-eth-logo.png']
        ];
        
        foreach ($cryptos as $crypto) {
            $stmt = $conn->prepare("
                INSERT OR IGNORE INTO cryptocurrencies (symbol, name, logo_url) 
                VALUES (?, ?, ?)
            ");
            $stmt->execute($crypto);
        }
        
        echo "✅ Cryptocurrencies inserted\n";
    }
    
    private function createAdminUser() {
        echo "👤 Creating admin user...\n";
        
        $conn = $this->db->getConnection();
        
        // Check if admin already exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute(['admin@yellowcrypto.com']);
        
        if ($stmt->fetch()) {
            echo "ℹ️  Admin user already exists\n";
            return;
        }
        
        // Create admin user
        $password_hash = password_hash('admin123', PASSWORD_BCRYPT, ['cost' => 12]);
        
        $stmt = $conn->prepare("
            INSERT INTO users (email, phone, password_hash, first_name, last_name, country) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            'admin@yellowcrypto.com',
            '+256700000000',
            $password_hash,
            'Admin',
            'User',
            'UG'
        ]);
        
        echo "✅ Admin user created (email: admin@yellowcrypto.com, password: admin123)\n";
    }
    
    private function createTradingPairs() {
        echo "💱 Creating trading pairs...\n";
        echo "ℹ️  Trading pairs will be created dynamically\n";
    }
    
    private function updateMarketPrices() {
        echo "📈 Market prices will be updated via API\n";
    }
}

// Run setup if called directly
if (basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    $setup = new SQLiteSetup();
    $setup->setup();
}
?>

