<?php
/**
 * Cron Job Script for Automated Balance Sync
 * Run this every 5 minutes to keep balances updated
 */

require_once 'api/blockchain_balance_sync.php';

// Set time limit for cron job
set_time_limit(300); // 5 minutes max

// Log the sync attempt
$logFile = __DIR__ . '/logs/balance_sync.log';
$logDir = dirname($logFile);

if (!is_dir($logDir)) {
    mkdir($logDir, 0755, true);
}

$logMessage = "[" . date('Y-m-d H:i:s') . "] Starting automated balance sync\n";
file_put_contents($logFile, $logMessage, FILE_APPEND);

try {
    $sync = new BlockchainBalanceSync();
    $result = $sync->syncAllBalances();
    
    if ($result['success']) {
        $logMessage = "[" . date('Y-m-d H:i:s') . "] Sync completed: ";
        $logMessage .= "{$result['updated_wallets']}/{$result['total_wallets']} wallets updated\n";
        
        // Log individual updates
        foreach ($result['results'] as $walletResult) {
            if ($walletResult['success'] && isset($walletResult['updated']) && $walletResult['updated']) {
                $logMessage .= "[" . date('Y-m-d H:i:s') . "] Updated: ";
                $logMessage .= "{$walletResult['currency']} {$walletResult['network']}: ";
                $logMessage .= "{$walletResult['old_balance']} → {$walletResult['new_balance']}\n";
            }
        }
    } else {
        $logMessage = "[" . date('Y-m-d H:i:s') . "] Sync failed: " . $result['error'] . "\n";
    }
    
    file_put_contents($logFile, $logMessage, FILE_APPEND);
    
} catch (Exception $e) {
    $logMessage = "[" . date('Y-m-d H:i:s') . "] Sync error: " . $e->getMessage() . "\n";
    file_put_contents($logFile, $logMessage, FILE_APPEND);
}
