<?php
/**
 * Alternative Blockchain API Testing
 * Try different approaches and endpoints
 */

require_once 'config/env.php';

echo "🔄 Alternative API Testing\n";
echo "==========================\n\n";

// Load environment configuration
EnvConfig::load();

$etherscanKey = EnvConfig::get('ETHERSCAN_API_KEY');
$bscscanKey = EnvConfig::get('BSCSCAN_API_KEY');
$polygonscanKey = EnvConfig::get('POLYGONSCAN_API_KEY');

$testAddress = '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6';

echo "🔍 Test 1: Try different V2 endpoint formats\n";
echo "===========================================\n";

// Try different V2 formats
$v2Formats = [
    'Standard V2' => "https://api.etherscan.io/v2?module=account&action=balance&address=$testAddress&tag=latest&apikey=$etherscanKey",
    'V2 with /api' => "https://api.etherscan.io/api/v2?module=account&action=balance&address=$testAddress&tag=latest&apikey=$etherscanKey",
    'V2 different format' => "https://api.etherscan.io/v2/api?module=account&action=balance&address=$testAddress&tag=latest&apikey=$etherscanKey"
];

foreach ($v2Formats as $name => $url) {
    echo "\n📡 Testing: $name\n";
    echo "URL: $url\n";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; BlockchainTest/1.0)');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    echo "HTTP Code: $httpCode\n";
    echo "Error: " . ($error ?: 'None') . "\n";
    
    if ($response !== false) {
        $data = json_decode($response, true);
        echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
        
        if (isset($data['result']) && $data['status'] === '1') {
            $balance = hexdec($data['result']) / pow(10, 18);
            echo "✅ SUCCESS - ETH Balance: $balance ETH\n";
        } else {
            echo "❌ FAILED - " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ REQUEST FAILED\n";
    }
}

echo "\n🔍 Test 2: Try without API key (public endpoints)\n";
echo "=================================================\n";

// Try public endpoints that might not require API key
$publicEndpoints = [
    'ETH Supply' => "https://api.etherscan.io/api?module=stats&action=ethsupply",
    'Latest Block' => "https://api.etherscan.io/api?module=proxy&action=eth_blockNumber",
    'Gas Oracle' => "https://api.etherscan.io/api?module=gastracker&action=gasoracle"
];

foreach ($publicEndpoints as $name => $url) {
    echo "\n📡 Testing: $name\n";
    echo "URL: $url\n";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; BlockchainTest/1.0)');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    echo "HTTP Code: $httpCode\n";
    echo "Error: " . ($error ?: 'None') . "\n";
    
    if ($response !== false) {
        $data = json_decode($response, true);
        echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
        
        if (isset($data['result']) && $data['status'] === '1') {
            echo "✅ SUCCESS\n";
        } else {
            echo "❌ FAILED - " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ REQUEST FAILED\n";
    }
}

echo "\n🔍 Test 3: Try different blockchain APIs\n";
echo "========================================\n";

// Try BSC and Polygon which might have different behavior
$otherApis = [
    'BSC Latest Block' => "https://api.bscscan.com/api?module=proxy&action=eth_blockNumber",
    'Polygon Latest Block' => "https://api.polygonscan.com/api?module=proxy&action=eth_blockNumber",
    'BSC Balance' => "https://api.bscscan.com/api?module=account&action=balance&address=$testAddress&tag=latest&apikey=$bscscanKey",
    'Polygon Balance' => "https://api.polygonscan.com/api?module=account&action=balance&address=$testAddress&tag=latest&apikey=$polygonscanKey"
];

foreach ($otherApis as $name => $url) {
    echo "\n📡 Testing: $name\n";
    echo "URL: $url\n";
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; BlockchainTest/1.0)');
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    echo "HTTP Code: $httpCode\n";
    echo "Error: " . ($error ?: 'None') . "\n";
    
    if ($response !== false) {
        $data = json_decode($response, true);
        echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
        
        if (isset($data['result']) && $data['status'] === '1') {
            if (strpos($name, 'Balance') !== false) {
                $balance = hexdec($data['result']) / pow(10, 18);
                echo "✅ SUCCESS - Balance: $balance\n";
            } else {
                echo "✅ SUCCESS\n";
            }
        } else {
            echo "❌ FAILED - " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ REQUEST FAILED\n";
    }
}

echo "\n🎯 Test 4: Try alternative blockchain data sources\n";
echo "=================================================\n";

// Try alternative sources like Alchemy, Infura, or public RPC endpoints
$alternativeSources = [
    'Alchemy ETH Balance' => "https://eth-mainnet.g.alchemy.com/v2/demo",
    'Public ETH RPC' => "https://cloudflare-eth.com",
    'Infura Public' => "https://mainnet.infura.io/v3/9aa3d95b3bc440fa88ea12eaa4456161"
];

foreach ($alternativeSources as $name => $url) {
    echo "\n📡 Testing: $name\n";
    echo "URL: $url\n";
    
    // For RPC endpoints, we need to send JSON-RPC requests
    if (strpos($url, 'alchemy') !== false || strpos($url, 'infura') !== false || strpos($url, 'cloudflare') !== false) {
        $payload = json_encode([
            'jsonrpc' => '2.0',
            'method' => 'eth_getBalance',
            'params' => [$testAddress, 'latest'],
            'id' => 1
        ]);
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; BlockchainTest/1.0)');
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        echo "HTTP Code: $httpCode\n";
        echo "Error: " . ($error ?: 'None') . "\n";
        
        if ($response !== false) {
            $data = json_decode($response, true);
            echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
            
            if (isset($data['result']) && !isset($data['error'])) {
                $balance = hexdec($data['result']) / pow(10, 18);
                echo "✅ SUCCESS - ETH Balance: $balance ETH\n";
            } else {
                echo "❌ FAILED - " . ($data['error']['message'] ?? 'Unknown error') . "\n";
            }
        } else {
            echo "❌ REQUEST FAILED\n";
        }
    }
}

echo "\n🎉 Alternative API Testing Completed!\n";
?>

