<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);

echo "Starting API debug test...\n";

try {
    require_once 'config/database.php';
    $db = new Database();
    echo "Database loaded\n";
    
    // Test the exact same logic as the API
    $input = [
        'wallet_id' => 1,
        'recipient' => '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6',
        'amount' => 0.1,
        'currency' => 'USDT',
        'network' => 'erc20'
    ];
    
    $wallet_id = intval($input['wallet_id'] ?? 0);
    $recipient = $input['recipient'] ?? '';
    $amount = floatval($input['amount'] ?? 0);
    $currency = $input['currency'] ?? '';
    $network = $input['network'] ?? '';
    
    echo "Parsed input:\n";
    echo "wallet_id: $wallet_id\n";
    echo "recipient: $recipient\n";
    echo "amount: $amount\n";
    echo "currency: $currency\n";
    echo "network: $network\n";
    
    if (!$wallet_id || !$recipient || $amount <= 0 || !$currency || !$network) {
        echo "ERROR: Missing required fields\n";
        exit;
    }
    
    // Test address validation
    echo "\nTesting address validation...\n";
    
    // Define the validation function locally
    function validateAddressFormat($address, $network) {
        $patterns = [
            'erc20' => '/^0x[a-fA-F0-9]{40}$/',
            'bep20' => '/^0x[a-fA-F0-9]{40}$/',
            'trc20' => '/^T[A-Za-z1-9]{33}$/',
            'polygon' => '/^0x[a-fA-F0-9]{40}$/',
            'base' => '/^0x[a-fA-F0-9]{40}$/',
            'arbitrum' => '/^0x[a-fA-F0-9]{40}$/',
            'optimism' => '/^0x[a-fA-F0-9]{40}$/',
            'solana' => '/^[1-9A-HJ-NP-Za-km-z]{32,44}$/',
            'btc' => '/^[13][a-km-zA-HJ-NP-Z1-9]{25,34}$|^bc1[a-z0-9]{39,59}$/'
        ];
        
        $pattern = $patterns[strtolower($network)] ?? null;
        return $pattern ? preg_match($pattern, $address) : false;
    }
    
    function calculateNetworkFees($network, $currency, $amount) {
        $feeRates = [
            'erc20' => ['gas_fee' => 0.01, 'gas_limit' => 21000],
            'bep20' => ['gas_fee' => 0.001, 'gas_limit' => 21000],
            'trc20' => ['gas_fee' => 0.001, 'gas_limit' => 1],
            'polygon' => ['gas_fee' => 0.0001, 'gas_limit' => 21000],
            'base' => ['gas_fee' => 0.0001, 'gas_limit' => 21000],
            'arbitrum' => ['gas_fee' => 0.0005, 'gas_limit' => 21000],
            'optimism' => ['gas_fee' => 0.0005, 'gas_limit' => 21000],
            'solana' => ['gas_fee' => 0.000005, 'gas_limit' => 1],
            'btc' => ['gas_fee' => 0.0001, 'gas_limit' => 1]
        ];
        
        $networkLower = strtolower($network);
        $baseFees = $feeRates[$networkLower] ?? ['gas_fee' => 0.01, 'gas_limit' => 21000];
        
        return [
            'gas_fee' => $baseFees['gas_fee'],
            'gas_limit' => $baseFees['gas_limit'],
            'estimated_total' => $baseFees['gas_fee']
        ];
    }
    
    $isValid = validateAddressFormat($recipient, $network);
    echo "Address valid: " . ($isValid ? 'yes' : 'no') . "\n";
    
    if (!$isValid) {
        echo "ERROR: Invalid address format\n";
        exit;
    }
    
    // Test wallet lookup
    echo "\nTesting wallet lookup...\n";
    $stmt = $db->getConnection()->prepare("
        SELECT id, user_id, currency, network, balance, address, private_key_encrypted 
        FROM crypto_wallets 
        WHERE id = ? AND user_id = ?
    ");
    $stmt->execute([$wallet_id, 1]);
    $userWallet = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$userWallet) {
        echo "ERROR: Wallet not found\n";
        exit;
    }
    
    echo "Wallet found:\n";
    print_r($userWallet);
    
    if ($userWallet['balance'] < $amount) {
        echo "ERROR: Insufficient balance\n";
        exit;
    }
    
    // Test fee calculation
    echo "\nTesting fee calculation...\n";
    $networkFees = calculateNetworkFees($network, $currency, $amount);
    echo "Network fees: " . json_encode($networkFees) . "\n";
    
    $totalCost = $amount + $networkFees['gas_fee'];
    echo "Total cost: $totalCost\n";
    
    if ($userWallet['balance'] < $totalCost) {
        echo "ERROR: Insufficient balance for amount + fees\n";
        exit;
    }
    
    echo "\nAll validations passed! Testing blockchain transaction...\n";
    
    // Test blockchain transaction
    require_once 'api/blockchain_manager.php';
    $blockchainManager = new BlockchainManager($db);
    
    $blockchainResult = $blockchainManager->sendRealTransaction(
        $userWallet['address'],
        $recipient,
        $amount,
        $currency,
        $network,
        $userWallet['private_key_encrypted']
    );
    
    echo "Blockchain result: " . json_encode($blockchainResult) . "\n";
    
} catch (Exception $e) {
    echo "Exception: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
} catch (Error $e) {
    echo "Fatal Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}

echo "Debug test completed.\n";

?>
