<?php
/**
 * YELLOW Crypto Exchange - Live Site Feature Testing
 * Test all features on https://kako.exchange/
 */

echo "🚀 YELLOW Crypto Exchange - Live Site Testing\n";
echo "============================================\n";
echo "Testing: https://kako.exchange/\n\n";

// Test configuration
$base_url = 'https://kako.exchange';
$api_url = $base_url . '/api';

// Function to make HTTP requests
function makeRequest($url, $method = 'GET', $data = null, $headers = []) {
    $ch = curl_init();
    
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    if ($method === 'POST') {
        curl_setopt($ch, CURLOPT_POST, true);
        if ($data) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            $headers[] = 'Content-Type: application/json';
        }
    }
    
    if (!empty($headers)) {
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    }
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    return [
        'response' => $response,
        'http_code' => $http_code,
        'error' => $error
    ];
}

// Test 1: Main Site Accessibility
echo "1️⃣ Testing Main Site Accessibility\n";
echo "----------------------------------\n";

$result = makeRequest($base_url);
if ($result['http_code'] === 200) {
    echo "✅ Main site is accessible (HTTP {$result['http_code']})\n";
    if (strpos($result['response'], 'YELLOW') !== false) {
        echo "✅ YELLOW branding found on site\n";
    }
    if (strpos($result['response'], 'Crypto Exchange') !== false) {
        echo "✅ Crypto Exchange text found\n";
    }
} else {
    echo "❌ Main site not accessible (HTTP {$result['http_code']})\n";
}

echo "\n";

// Test 2: API Health Check
echo "2️⃣ Testing API Health Check\n";
echo "----------------------------\n";

$result = makeRequest($api_url . '/system/health-check');
if ($result['http_code'] === 200) {
    echo "✅ API health check successful (HTTP {$result['http_code']})\n";
    $data = json_decode($result['response'], true);
    if ($data) {
        echo "✅ API returned valid JSON response\n";
        if (isset($data['status'])) {
            echo "✅ API status: " . $data['status'] . "\n";
        }
    }
} else {
    echo "❌ API health check failed (HTTP {$result['http_code']})\n";
    if ($result['error']) {
        echo "❌ Error: " . $result['error'] . "\n";
    }
}

echo "\n";

// Test 3: Authentication Endpoints
echo "3️⃣ Testing Authentication Endpoints\n";
echo "------------------------------------\n";

// Test login endpoint (without credentials)
$result = makeRequest($api_url . '/auth/login', 'POST', ['email' => 'test@test.com', 'password' => 'test']);
if ($result['http_code'] === 401 || $result['http_code'] === 400) {
    echo "✅ Login endpoint responding correctly (HTTP {$result['http_code']})\n";
} else {
    echo "❌ Login endpoint unexpected response (HTTP {$result['http_code']})\n";
}

// Test registration endpoint
$result = makeRequest($api_url . '/auth/register', 'POST', [
    'email' => 'test@test.com',
    'password' => 'test123',
    'first_name' => 'Test',
    'last_name' => 'User'
]);
if ($result['http_code'] === 400 || $result['http_code'] === 409) {
    echo "✅ Registration endpoint responding correctly (HTTP {$result['http_code']})\n";
} else {
    echo "❌ Registration endpoint unexpected response (HTTP {$result['http_code']})\n";
}

echo "\n";

// Test 4: Market Data Endpoints
echo "4️⃣ Testing Market Data Endpoints\n";
echo "---------------------------------\n";

// Test market overview
$result = makeRequest($api_url . '/market/overview');
if ($result['http_code'] === 200) {
    echo "✅ Market overview endpoint working (HTTP {$result['http_code']})\n";
    $data = json_decode($result['response'], true);
    if ($data && isset($data['cryptocurrencies'])) {
        echo "✅ Market data returned cryptocurrencies\n";
    }
} else {
    echo "❌ Market overview endpoint failed (HTTP {$result['http_code']})\n";
}

// Test crypto prices
$result = makeRequest($api_url . '/crypto/prices');
if ($result['http_code'] === 200) {
    echo "✅ Crypto prices endpoint working (HTTP {$result['http_code']})\n";
} else {
    echo "❌ Crypto prices endpoint failed (HTTP {$result['http_code']})\n";
}

echo "\n";

// Test 5: Trading Endpoints
echo "5️⃣ Testing Trading Endpoints\n";
echo "-----------------------------\n";

// Test trading pairs
$result = makeRequest($api_url . '/trading/pairs');
if ($result['http_code'] === 200) {
    echo "✅ Trading pairs endpoint working (HTTP {$result['http_code']})\n";
    $data = json_decode($result['response'], true);
    if ($data && isset($data['pairs'])) {
        echo "✅ Trading pairs data returned\n";
    }
} else {
    echo "❌ Trading pairs endpoint failed (HTTP {$result['http_code']})\n";
}

// Test market prices
$result = makeRequest($api_url . '/trading/prices');
if ($result['http_code'] === 200) {
    echo "✅ Trading prices endpoint working (HTTP {$result['http_code']})\n";
} else {
    echo "❌ Trading prices endpoint failed (HTTP {$result['http_code']})\n";
}

echo "\n";

// Test 6: Wallet Endpoints
echo "6️⃣ Testing Wallet Endpoints\n";
echo "----------------------------\n";

// Test wallet endpoints (should require authentication)
$result = makeRequest($api_url . '/user/wallets');
if ($result['http_code'] === 401) {
    echo "✅ Wallet endpoints properly protected (HTTP {$result['http_code']})\n";
} else {
    echo "❌ Wallet endpoints unexpected response (HTTP {$result['http_code']})\n";
}

echo "\n";

// Test 7: Admin Panel
echo "7️⃣ Testing Admin Panel\n";
echo "----------------------\n";

$result = makeRequest($base_url . '/admin/');
if ($result['http_code'] === 200) {
    echo "✅ Admin panel accessible (HTTP {$result['http_code']})\n";
    if (strpos($result['response'], 'admin') !== false) {
        echo "✅ Admin panel content found\n";
    }
} else {
    echo "❌ Admin panel not accessible (HTTP {$result['http_code']})\n";
}

echo "\n";

// Test 8: Key Pages
echo "8️⃣ Testing Key Pages\n";
echo "--------------------\n";

$pages = [
    'login.html' => 'Login Page',
    'sign_up.html' => 'Sign Up Page',
    'user_dashboard.html' => 'User Dashboard',
    'market_overview.html' => 'Market Overview',
    'buy.html' => 'Buy Page',
    'sell.html' => 'Sell Page',
    'crypto_wallet.html' => 'Crypto Wallet',
    'transaction_history.html' => 'Transaction History'
];

foreach ($pages as $page => $name) {
    $result = makeRequest($base_url . '/' . $page);
    if ($result['http_code'] === 200) {
        echo "✅ {$name} accessible (HTTP {$result['http_code']})\n";
    } else {
        echo "❌ {$name} not accessible (HTTP {$result['http_code']})\n";
    }
}

echo "\n";

// Test 9: Blockchain Service (if accessible)
echo "9️⃣ Testing Blockchain Service\n";
echo "------------------------------\n";

// Try to access blockchain service (might be internal)
$result = makeRequest('http://localhost:3001/health');
if ($result['http_code'] === 200) {
    echo "✅ Blockchain service accessible locally\n";
} else {
    echo "ℹ️  Blockchain service not accessible locally (expected for production)\n";
}

echo "\n";

// Summary
echo "📊 TESTING SUMMARY\n";
echo "==================\n";
echo "✅ Main site: https://kako.exchange/\n";
echo "✅ API base: https://kako.exchange/api/\n";
echo "✅ Database: kakoexchange25_kakoexchange\n";
echo "✅ All key pages tested\n";
echo "✅ API endpoints tested\n";
echo "\n";
echo "🎯 Next Steps:\n";
echo "1. Test user registration and login\n";
echo "2. Test trading functionality\n";
echo "3. Test wallet operations\n";
echo "4. Test payment integration\n";
echo "5. Test admin panel functionality\n";
echo "\n";
echo "🚀 YELLOW Crypto Exchange is ready for comprehensive testing!\n";
?>
