<?php
/**
 * REAL BLOCKCHAIN INTEGRATION TEST
 * This script tests that all components are using real blockchain data
 */

require_once __DIR__ . '/config/database.php';
require_once __DIR__ . '/api/wallet_generator.php';
require_once __DIR__ . '/api/wallet.php';

class RealBlockchainTest {
    private $db;
    
    public function __construct() {
        $this->db = new Database();
    }
    
    public function runAllTests() {
        echo "🚀 REAL BLOCKCHAIN INTEGRATION TEST\n";
        echo "=====================================\n\n";
        
        $this->testWalletGeneration();
        $this->testBlockchainService();
        $this->testBalanceChecking();
        $this->testTransactionProcessing();
        
        echo "\n✅ ALL REAL BLOCKCHAIN TESTS COMPLETED!\n";
    }
    
    private function testWalletGeneration() {
        echo "🔍 Testing REAL Wallet Generation...\n";
        
        $walletGenerator = new WalletGenerator($this->db->getConnection());
        
        // Test Ethereum wallet generation
        $ethWallet = $walletGenerator->generateWallet(1, 'erc20', 'USDT');
        if ($ethWallet['success']) {
            $address = $ethWallet['wallet']['address'];
            echo "✅ Ethereum wallet generated: {$address}\n";
            echo "   Format check: " . (preg_match('/^0x[a-fA-F0-9]{40}$/', $address) ? "PASS" : "FAIL") . "\n";
        } else {
            echo "❌ Ethereum wallet generation failed\n";
        }
        
        // Test Tron wallet generation
        $tronWallet = $walletGenerator->generateWallet(1, 'trc20', 'USDT');
        if ($tronWallet['success']) {
            $address = $tronWallet['wallet']['address'];
            echo "✅ Tron wallet generated: {$address}\n";
            echo "   Format check: " . (preg_match('/^T[a-zA-Z0-9]{33}$/', $address) ? "PASS" : "FAIL") . "\n";
        } else {
            echo "❌ Tron wallet generation failed\n";
        }
        
        // Test Solana wallet generation
        $solWallet = $walletGenerator->generateWallet(1, 'solana', 'USDT');
        if ($solWallet['success']) {
            $address = $solWallet['wallet']['address'];
            echo "✅ Solana wallet generated: {$address}\n";
            echo "   Format check: " . (strlen($address) >= 32 && strlen($address) <= 44 ? "PASS" : "FAIL") . "\n";
        } else {
            echo "❌ Solana wallet generation failed\n";
        }
        
        echo "\n";
    }
    
    private function testBlockchainService() {
        echo "🔍 Testing Blockchain Service Connection...\n";
        
        $url = 'http://localhost:3001/balance';
        
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_CONNECTTIMEOUT => 5
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        if ($httpCode === 200 && $response) {
            $data = json_decode($response, true);
            if ($data && isset($data['success'])) {
                echo "✅ Blockchain service is running\n";
                echo "   Address: {$data['address']}\n";
                echo "   TRX Balance: {$data['trxBalance']}\n";
                echo "   USDT Balance: {$data['usdtBalance']}\n";
            } else {
                echo "❌ Blockchain service returned invalid response\n";
            }
        } else {
            echo "❌ Blockchain service is not running (HTTP {$httpCode})\n";
            echo "   Please start the blockchain service: node blockchain-service.js\n";
        }
        
        echo "\n";
    }
    
    private function testBalanceChecking() {
        echo "🔍 Testing REAL Balance Checking...\n";
        
        // Test with real addresses
        $testAddresses = [
            'ethereum' => '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6',
            'tron' => 'TQn9Y2khEsLJW1ChVWFMSMeRDow5KcbLSE',
            'bsc' => '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6'
        ];
        
        foreach ($testAddresses as $network => $address) {
            echo "   Testing {$network} balance for {$address}...\n";
            
            $url = "http://localhost:8000/api/blockchain/balance?network={$network}&address={$address}&token=USDT";
            
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_CONNECTTIMEOUT => 10
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            
            if ($httpCode === 200 && $response) {
                $data = json_decode($response, true);
                if ($data && isset($data['success'])) {
                    echo "   ✅ {$network}: {$data['balance']} {$data['currency']}\n";
                } else {
                    echo "   ❌ {$network}: API error - " . ($data['error'] ?? 'Unknown error') . "\n";
                }
            } else {
                echo "   ❌ {$network}: HTTP {$httpCode} - Service unavailable\n";
            }
        }
        
        echo "\n";
    }
    
    private function testTransactionProcessing() {
        echo "🔍 Testing REAL Transaction Processing...\n";
        
        $walletAPI = new WalletAPI(1);
        
        // Test transaction data
        $transactionData = [
            'crypto_id' => 1, // USDT
            'network_id' => 2, // TRC20
            'amount' => 1.0,
            'to_address' => 'TQn9Y2khEsLJW1ChVWFMSMeRDow5KcbLSE',
            'fee' => 0.1
        ];
        
        echo "   Testing transaction: {$transactionData['amount']} USDT to {$transactionData['to_address']}\n";
        
        // This would normally process a real transaction
        // For testing, we'll just verify the data structure
        if (isset($transactionData['crypto_id']) && isset($transactionData['network_id']) && 
            isset($transactionData['amount']) && isset($transactionData['to_address'])) {
            echo "   ✅ Transaction data structure is valid\n";
            echo "   ✅ All required fields present\n";
            echo "   ✅ Address format: " . (preg_match('/^T[a-zA-Z0-9]{33}$/', $transactionData['to_address']) ? "VALID" : "INVALID") . "\n";
        } else {
            echo "   ❌ Transaction data structure is invalid\n";
        }
        
        echo "\n";
    }
}

// Run the tests
if (basename(__FILE__) === basename($_SERVER['SCRIPT_NAME'])) {
    $test = new RealBlockchainTest();
    $test->runAllTests();
}
?>