<?php
/**
 * Test Updated Blockchain APIs
 * Test the updated APIs with working approaches
 */

require_once 'config/database.php';
require_once 'api/blockchain_apis.php';

echo "🚀 Updated Blockchain API Tests\n";
echo "===============================\n\n";

try {
    $db = new Database();
    $conn = $db->getConnection();
    $blockchainAPIs = new BlockchainAPIs($conn);
    
    $testAddress = '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6';
    
    echo "💰 Test 1: Balance Retrieval with Updated APIs\n";
    echo "===============================================\n";
    
    $networks = ['ethereum', 'bsc', 'polygon'];
    foreach ($networks as $network) {
        echo "🔍 Getting balance for $network address: $testAddress\n";
        
        try {
            $balance = $blockchainAPIs->getAccountBalance($network, $testAddress);
            if ($balance['success']) {
                echo "✅ $network balance: " . $balance['balance'] . " " . $balance['currency'] . "\n";
            } else {
                echo "❌ $network balance failed: " . $balance['error'] . "\n";
            }
        } catch (Exception $e) {
            echo "❌ $network balance exception: " . $e->getMessage() . "\n";
        }
        echo "\n";
    }
    
    echo "🔍 Test 2: Address Validation\n";
    echo "=============================\n";
    
    $testAddresses = [
        ['ethereum', '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6'],
        ['ethereum', '0xinvalid'],
        ['bsc', '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6'],
        ['polygon', '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6']
    ];
    
    foreach ($testAddresses as $test) {
        list($network, $address) = $test;
        echo "🔍 Validating $network address: $address\n";
        
        try {
            $validation = $blockchainAPIs->validateAddress($network, $address);
            if ($validation['success']) {
                $result = $validation['valid'] ? 'Valid' : 'Invalid';
                echo "✅ $network address validation: $result\n";
            } else {
                echo "❌ $network address validation failed: " . $validation['error'] . "\n";
            }
        } catch (Exception $e) {
            echo "❌ $network address validation exception: " . $e->getMessage() . "\n";
        }
        echo "\n";
    }
    
    echo "⛽ Test 3: Gas Prices\n";
    echo "====================\n";
    
    foreach ($networks as $network) {
        echo "⛽ Getting gas prices for $network...\n";
        
        try {
            $gasPrices = $blockchainAPIs->getGasPrices($network);
            if ($gasPrices['success']) {
                echo "✅ $network gas prices:\n";
                echo "   Slow: " . $gasPrices['slow'] . " Gwei\n";
                echo "   Standard: " . $gasPrices['standard'] . " Gwei\n";
                echo "   Fast: " . $gasPrices['fast'] . " Gwei\n";
            } else {
                echo "❌ $network gas prices failed: " . $gasPrices['error'] . "\n";
            }
        } catch (Exception $e) {
            echo "❌ $network gas prices exception: " . $e->getMessage() . "\n";
        }
        echo "\n";
    }
    
    echo "📜 Test 4: Transaction History\n";
    echo "==============================\n";
    
    foreach ($networks as $network) {
        echo "📜 Getting transaction history for $network address: $testAddress\n";
        
        try {
            $history = $blockchainAPIs->getTransactionHistory($network, $testAddress, 5);
            if ($history['success']) {
                echo "✅ $network transaction history retrieved:\n";
                echo "   Total transactions: " . count($history['transactions']) . "\n";
                if (!empty($history['transactions'])) {
                    $latest = $history['transactions'][0];
                    echo "   Latest transaction:\n";
                    echo "     Hash: " . substr($latest['hash'], 0, 20) . "...\n";
                    echo "     Block: " . $latest['blockNumber'] . "\n";
                    echo "     Value: " . $latest['value'] . "\n";
                }
            } else {
                echo "❌ $network transaction history failed: " . $history['error'] . "\n";
            }
        } catch (Exception $e) {
            echo "❌ $network transaction history exception: " . $e->getMessage() . "\n";
        }
        echo "\n";
    }
    
    echo "🎯 Test 5: Token Balance (USDT)\n";
    echo "===============================\n";
    
    foreach ($networks as $network) {
        echo "🪙 Getting USDT balance for $network address: $testAddress\n";
        
        try {
            $tokenBalance = $blockchainAPIs->getAccountBalance($network, $testAddress, 'USDT');
            if ($tokenBalance['success']) {
                echo "✅ $network USDT balance: " . $tokenBalance['balance'] . " USDT\n";
            } else {
                echo "❌ $network USDT balance failed: " . $tokenBalance['error'] . "\n";
            }
        } catch (Exception $e) {
            echo "❌ $network USDT balance exception: " . $e->getMessage() . "\n";
        }
        echo "\n";
    }
    
    echo "🎉 Updated Blockchain API Tests Completed!\n";
    echo "==========================================\n";
    echo "✅ All tests have been executed with updated blockchain APIs\n";
    echo "📊 Check the results above to verify API functionality\n";
    
} catch (Exception $e) {
    echo "❌ Test Error: " . $e->getMessage() . "\n";
    echo "Stack trace: " . $e->getTraceAsString() . "\n";
}
?>

