<?php
/**
 * Test Working Blockchain APIs
 * Test with different approaches to find working endpoints
 */

require_once 'config/env.php';

echo "🔧 Working API Testing\n";
echo "======================\n\n";

// Load environment configuration
EnvConfig::load();

$etherscanKey = EnvConfig::get('ETHERSCAN_API_KEY');
$bscscanKey = EnvConfig::get('BSCSCAN_API_KEY');
$polygonscanKey = EnvConfig::get('POLYGONSCAN_API_KEY');

$testAddress = '0x742d35Cc6634C0532925a3b8D4C9db96C4b4d8b6';

echo "🔍 Test 1: Etherscan with different endpoint formats\n";
echo "===================================================\n";

// Test different endpoint formats
$endpoints = [
    'V1 Standard' => "https://api.etherscan.io/api?module=account&action=balance&address=$testAddress&tag=latest&apikey=$etherscanKey",
    'V1 No tag' => "https://api.etherscan.io/api?module=account&action=balance&address=$testAddress&apikey=$etherscanKey",
    'V2 Standard' => "https://api.etherscan.io/v2?module=account&action=balance&address=$testAddress&tag=latest&apikey=$etherscanKey",
    'V2 No tag' => "https://api.etherscan.io/v2?module=account&action=balance&address=$testAddress&apikey=$etherscanKey"
];

foreach ($endpoints as $name => $url) {
    echo "\n📡 Testing: $name\n";
    echo "URL: $url\n";
    
    $context = stream_context_create([
        'http' => [
            'timeout' => 10,
            'user_agent' => 'Mozilla/5.0 (compatible; BlockchainTest/1.0)'
        ]
    ]);
    
    $response = file_get_contents($url, false, $context);
    if ($response !== false) {
        $data = json_decode($response, true);
        echo "Status: " . ($data['status'] ?? 'unknown') . "\n";
        echo "Message: " . ($data['message'] ?? 'none') . "\n";
        
        if (isset($data['result']) && $data['status'] === '1') {
            $balance = hexdec($data['result']) / pow(10, 18);
            echo "✅ SUCCESS - ETH Balance: $balance ETH\n";
        } else {
            echo "❌ FAILED - " . ($data['message'] ?? 'Unknown error') . "\n";
        }
    } else {
        echo "❌ REQUEST FAILED\n";
    }
}

echo "\n🔍 Test 2: Try with a known active address\n";
echo "==========================================\n";

// Try with a well-known address that should have activity
$knownAddress = '0xd8dA6BF26964aF9D7eEd9e03E53415D37aA96045'; // Vitalik's address
$url = "https://api.etherscan.io/api?module=account&action=balance&address=$knownAddress&tag=latest&apikey=$etherscanKey";

echo "Testing with Vitalik's address: $knownAddress\n";
echo "URL: $url\n";

$context = stream_context_create([
    'http' => [
        'timeout' => 10,
        'user_agent' => 'Mozilla/5.0 (compatible; BlockchainTest/1.0)'
    ]
]);

$response = file_get_contents($url, false, $context);
if ($response !== false) {
    $data = json_decode($response, true);
    echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
    
    if (isset($data['result']) && $data['status'] === '1') {
        $balance = hexdec($data['result']) / pow(10, 18);
        echo "✅ SUCCESS - ETH Balance: $balance ETH\n";
    } else {
        echo "❌ FAILED - " . ($data['message'] ?? 'Unknown error') . "\n";
    }
} else {
    echo "❌ REQUEST FAILED\n";
}

echo "\n🔍 Test 3: Test API Key validity\n";
echo "===============================\n";

// Test if API key is valid by checking rate limit
$url = "https://api.etherscan.io/api?module=stats&action=ethsupply&apikey=$etherscanKey";
echo "Testing API key with ETH supply endpoint: $url\n";

$context = stream_context_create([
    'http' => [
        'timeout' => 10,
        'user_agent' => 'Mozilla/5.0 (compatible; BlockchainTest/1.0)'
    ]
]);

$response = file_get_contents($url, false, $context);
if ($response !== false) {
    $data = json_decode($response, true);
    echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
    
    if (isset($data['result']) && $data['status'] === '1') {
        $supply = hexdec($data['result']) / pow(10, 18);
        echo "✅ API KEY VALID - ETH Supply: $supply ETH\n";
    } else {
        echo "❌ API KEY ISSUE - " . ($data['message'] ?? 'Unknown error') . "\n";
    }
} else {
    echo "❌ REQUEST FAILED\n";
}

echo "\n🔍 Test 4: Alternative approach with cURL\n";
echo "=========================================\n";

$url = "https://api.etherscan.io/api?module=account&action=balance&address=$testAddress&tag=latest&apikey=$etherscanKey";

$ch = curl_init();
curl_setopt($ch, CURLOPT_URL, $url);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_TIMEOUT, 10);
curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (compatible; BlockchainTest/1.0)');
curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);

$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$error = curl_error($ch);

curl_close($ch);

echo "cURL Test Results:\n";
echo "HTTP Code: $httpCode\n";
echo "Error: " . ($error ?: 'None') . "\n";

if ($response !== false) {
    $data = json_decode($response, true);
    echo "Response: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
    
    if (isset($data['result']) && $data['status'] === '1') {
        $balance = hexdec($data['result']) / pow(10, 18);
        echo "✅ SUCCESS - ETH Balance: $balance ETH\n";
    } else {
        echo "❌ FAILED - " . ($data['message'] ?? 'Unknown error') . "\n";
    }
} else {
    echo "❌ REQUEST FAILED\n";
}

echo "\n🎉 Working API Testing Completed!\n";
?>

